<?php
/**
 * Describes an experiment in which an user can apply different operations and then can observe the results
 * either by inspecting the resulting file or by downloading the result analysis
 * @author andrei
 *
 */
class Experiment extends CActiveRecord
{
    public static function model($className=__CLASS__)
    {
        return parent::model($className);
    }
	
    public function tableName()
    {
    	return SqlUtil::EXPERIMENT_TBL;
    }
    
    public function primaryKey()
    {
    	return SqlUtil::ID_COL;
    	// For composite primary key, return an array like the following
    	// return array('pk1', 'pk2');
    }
    
    
    public $id;
    /** @var String title of the experiment*/
	public $title;
	public $date;
	public $description;
	/** @var String Full path of the location of the experiment files **/
	public $path;
	/** 
	 * @var FileItem The file on which are applied the operations; Users may upload multiple files
	 * but only one can be current
	 */
	public $currentOpFile;
	/**
	 * @var array of Operation An array of operations done over a certain file in which you save 
	 * each state of the file
	 */
	public $operationHistory;
	
	/** @var int The id of the user who is the owner of this experiment; In case the user is anonymous the id is -1; */
	public $id_user;
	/** @var int The id of the current op file stored in the database; Needed for auto operations of YII framework */
	public $id_fileitem;
	
	public function __construct($title = NULL,
			$date = NULL,
			$description = NULL,
			$path = NULL,
			$idUser = Util::ID_UNDEFINED,
			$currentOpFile = NULL,
			$id_fileitem = Util::ID_UNDEFINED,
			$operationHistory = array()) {
		$this->title = $title;
		$this->date = $date;
		$this->description = $description;
		$this->path = $path;
		$this->id_user = $idUser;
		$this->currentOpFile = $currentOpFile;
		$this->id_fileitem = $id_fileitem;
		$this->operationHistory = $operationHistory;
	}
	
	public function relations()
	{
		return array(
				'user'=>array(self::BELONGS_TO, 'LoginForm', 
						SqlUtil::EXPERIMENT_USER_FK),
				'fileitem'=>array(self::HAS_ONE, 'FileItem',
						SqlUtil::EXPERIMENT_FILEITEM_FK)
		);
	}
	
	public function attributeLabels()
	{
		return array('lblTitle'=>Yii::t('lstrings', 'TITLE'),
				'lblDescription'=>Yii::t('lstrings', 'DESCRIPTION'));
	}
	
	public function getIdFileItem() {
		return $this->id_fileitem;
	}
	
	public function getPath() {
		return $this->path;
	}
	
	/**
	 * Retrieve the current file for the experiment
	 * @return FileItem the current file on which the operations are applied
	 */
	public function getCurrentOpFile() {
		if ($this->currentOpFile != NULL && $this->currentOpFile->isVirtual()) 
			return NULL;
		else
			return $this->currentOpFile;
	}
	
	/**
	 * Set the current file for the experiment; Remember that this one will be used as input for any
	 * operation of the user
	 * @param FileItem $file the file to be set as the current for the experiment
	 */
	public function setCurrentOpFile(FileItem $file) {
		if (!Yii::app()->user->isGuest) {
			//check if the current file is virtual, and if yes remove it from db
			/* if ($this->currentOpFile !== NULL && $this->currentOpFile->isVirtual())
				FileItemFactory::deleteFileDb($this->currentOpFile->getId()); */
			//set the id for the experiment
			$this->id_fileitem = $file->getId();
			
		}
		$this->currentOpFile = $file;
		
	}
	
	/**
	 * Add an operation to the history list
	 * @param Operation $operation the operation to be added in the list
	 */
	public function addOperationHistory(Operation $operation) {
		$this->operationHistory[] = $operation;
	}
	
	/**
	 * Sets the list of operations as the value of the received parameter
	 * @param array Operation $newOperations The new list with the operations
	 */
	public function setOperationHistory(array $newOperations) {
			//for each operation, try to remove all data associated with it
			
			//set the new value
			$this->operationHistory = $newOperations;
	}
	
	/**
	 * Get the last operation in the queue or NULL if it is empty
	 */
	public function getLastOperation() {
		if (count($this->operationHistory) > 0)
			return $this->operationHistory[count($this->operationHistory)-1];
		else 
			return NULL;
	}
	
	public function getOperationHistory() {
		return $this->operationHistory;
	}
	
	public function getOpHistById($id) {
		foreach ($this->operationHistory as $op) {
			if (strcmp($id,$op->getId()) === 0)
				return $op;
		}
		return NULL;
	}
	
	/**
	 * Helper function to set the result of the last executed operation in the queue 
	 * if modifications are necessary; This is needed because each time you execute an operation,
	 * the old file acts as an input (it may be the result for the previous operation) and
	 * in return the new generated file will take the name and the old one will change its name;
	 * we only need to set the name as the size remains constant
	 * @param string $fname the new name on the disk
	 */
	public function setFNameLastOpQueue($fname) {
		if ($this->getLastOperation() !== NULL ){
			$f = $this->getLastOperation()->getResult()->getFile();
			if (isset($f) && $f->getType() === FileItemFactory::FILE_TYPE_DATA) 
				$f->setName($fname);
		}
	}
	
	public function getTitle() {
		return $this->title;
	}
	
	public function getDescription() {
		return $this->description;
	}
	
	public static function getAllExperimentsByUser() {
		$criteria = new CDbCriteria();
		$criteria->order = SqlUtil::EXPERIMENT_DATE_COL.' DESC';
// 		$criteria->condition = SqlUtil::EXPERIMENT_USER_FK.' = :id';
// 		$criteria->params = array(':id' => Yii::app()->user->getId());
		$criteria->compare(SqlUtil::EXPERIMENT_USER_FK, Yii::app()->user->getId(), TRUE, 'AND');
	
		$result = Experiment::model()->findAll($criteria);
		return new CArrayDataProvider($result, array());
	}
	
	public static function getExpLink() {
		return 'exp/experiment/page/view/exp_list';
	}
	
	public function getId() {
		return $this->id;
	}
	
	public function setId($value) {
		$this->id = $value;
	}
}