<?php

/**
 * Builds an object representing a file using the type of the file
 * and some additional information; hides the whole generation process 
 * @author andrei
 *
 */
class FileItemFactory {
	
	const FILE_TYPE_GENERIC = 3001;
	/** use this when you need a csv file containing the genes to be loaded and parsed for information */
	const FILE_TYPE_DATA = 3002;
	/** use this when you need a csv file containing the genes to be loaded and parsed for information */
	const FILE_TYPE_DATA_TR_RA = 3003;
	/** use this when you need a csv file containing the genes to be loaded and parsed for information */
	const FILE_TYPE_DATA_TIME_POINT = 3004;
	/** This type of file doesn't exist on the disk, it is there to fill the current file position */
	const FILE_TYPE_VIRTUAL = 3005;
	//const FILE_TYPE_NEWICK = 103;
	
	public static $DELIMITERS = array('\t', ';', ',', '|');
	
	public static function createFileVirtual() {
		return new FileItem(FileItemFactory::FILE_TYPE_VIRTUAL,
				'diskName', 'name', -1);
	}
	
	/**
	 * Removes a file from database; The current model of DB has a cascade action for update and del
	 * thus it will auto remove all necessary files from all tables; The exception is when the array
	 * of times is used, then being necessary to remove all the values manually
	 * @param int $idFile The id of the file to be removed
	 * @throws Exception Problems when trying to connect to DB
	 */
	public static function deleteFileDb($idFile) {
		try {
			//remove all entries for a file item from the array table
			$fileI =  Yii::app()->db->createCommand(SqlUtil::FILEITEMDATAARR_DELETE)->
				execute(array(/* ':arrtbl' => SqlUtil::FILEITEMDATAARR_TBL,
						':fileitemlnk' => SqlUtil::FILEITEMDATAARRLINK_TBL, 
						':arrtblID' => SqlUtil::ID_COL,
						':fileitemlnkarrID' => SqlUtil::FILEITEMDATAARRLINK_FILEITEMDATAARR_FK, 
						':fileitemlnkdataID' => SqlUtil::FILEITEMDATAARRLINK_FILEITEMDATA_FK,*/
						':fileID' => $idFile));
			//remove the key pointing to the file item data
			$fileI =  Yii::app()->db->createCommand()->delete(SqlUtil::FILEITEMDATAARRLINK_TBL,
					array('AND', SqlUtil::FILEITEMDATAARRLINK_FILEITEMDATA_FK.' = '.$idFile));
			
			$result = FileItem::model()->deleteByPk($idFile);
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	/**
	 * Special method which will load all the values from the database without checking them on the disk
	 * @param int $id The id of the file to be loaded from the database 
	 * @throws Exception Problems when trying to execute the DB statements
	 */
	public static function createFileDb($id) {
		//determine type of the file by searching in the database
		try {
			$f = new FileItem();
			$f->loadFile($id);
			switch ($f->getFileType()) {
				//the following 2 cases won't require any additional instantiation
				//case FileItemFactory::FILE_TYPE_VIRTUAL:
				//case FileItemFactory::FILE_TYPE_GENERIC:
				case FileItemFactory::FILE_TYPE_DATA_TR_RA:
					$f = new FileItemDataTRRA();
					$f->loadFile($id);
					break;
				case FileItemFactory::FILE_TYPE_DATA_TIME_POINT:
					$f = new FileItemDataTimePoint();
					$f->loadFile($id);
					break;
				case FileItemFactory::FILE_TYPE_DATA:
					$f = new FileItemData();
					$f->loadFile($id);
					break;
			}
			return $f;
		} catch(Exception $e) {
			throw $e;
		}
		
	}
	
	/**
	 * Creates a certain type of file as it is instructed by the configuration sent
	 * @param int $type the type of file to be created; FILE_TYPE_GENERIC, FILE_TYPE_DATA, FILE_TYPE_DATA_TR_RA
	 * 		FILE_TYPE_DATA_TIME_POINT, FILE_TYPE_VIRTUAL
	 * @param boolean $checkDuplicates Whether to check or not if the file has duplicate genes on the first column;
	 *    Used only when an user uploads a data file
	 * @param string $fullPath The full path on the disk without the name
	 * @param string $diskName The name on the disk
	 * @param string $name The virtual name, shown to the user
	 * @param int $id ID of the object
	 * @throws Exception
	 */
	public static function createFile($type, $fullPath, $diskName, $name, $size = NULL, 
			$id = Util::ID_UNDEFINED, $checkDuplicates = false)//, FileItemParams $params)
	{
		if ($type === FileItemFactory::FILE_TYPE_GENERIC) {
			$result = new FileItem(FileItemFactory::FILE_TYPE_GENERIC,
					$diskName, $name, ($size !== NULL ? $size : filesize($fullPath.$diskName)));
			
		} else if ($type === FileItemFactory::FILE_TYPE_DATA_TR_RA ||
				$type === FileItemFactory::FILE_TYPE_DATA_TIME_POINT ||
				// Data file type is used for the case when the user uploads a file 
				$type === FileItemFactory::FILE_TYPE_DATA) {
			ini_set('auto_detect_line_endings',TRUE);//detect the end of line delimiter
			if (($fp = fopen($fullPath.
					$diskName, "r")) !== FALSE) {
				$delim = NULL;//Yii::app()->session['currentExperiment']->getCurrentOpFile()->getDelimiter();
				$hasNA = false;
				
				//dtermine the delimiters
				$headerCSV = fgets($fp);
				foreach (FileItemFactory::$DELIMITERS as $delimOpt) {
					if (strpos($headerCSV, $delimOpt) !== false) {
						$delim = $delimOpt;
						break;
					}
				}
				if ($delim == NULL)
					throw new Exception(Yii::t('lstrings', '10005'), 10005);
				//move to the beggining of the file
				//rewind($fp);
				//try and read the header
				$record = explode($delim, $headerCSV);//fgetcsv($fp, 0, $delim);
				$len = count($record);
				
				//validate each cell and determine the total number of experiment times for each type
				//start from one since the first column contains the genes name
				//if ($type === FileItemFactory::FILE_TYPE_DATA_TR_RA) {
					$tr = 0;
					$ra = 0;
					$tp = 0;
					$arrTP = array();
					$arrTR = array();
					$arrRA = array();
					for ($index = 1; $index<$len; $index++) {
						$record[$index] = strtoupper(trim($record[$index]));
						if (preg_match(FileItemDataTRRA::TR_PATTERN, $record[$index])) {
							preg_match_all('/([\d]+)/', $record[$index], $match);
							$arrTR[$match[0][0]] =  $match[0][0];
							$tr++;
						} else if (preg_match(FileItemDataTRRA::RA_PATTERN, $record[$index])) {
							preg_match_all('/([\d]+)/', $record[$index], $match);
							$arrRA[$match[0][0]] =  $match[0][0];
							$ra++;
						} else if (preg_match(FileItemDataTimePoint::TIME_POINT_PATTERN, $record[$index])
								|| $record[$index] === FileItemDataTimePoint::TIME_POINT_PATTERN_FIRST_COL) {
							preg_match_all('/([\d]+)/', $record[$index], $match);
							$arrTP[$match[0][0]] =  $match[0][0];
							$tp++;
						} else
							throw new Exception(Yii::t('lstrings', 'ERR_FILE_HEADER_FORMAT'));
					}
				//} else {//support for FileItemFactory::FILE_TYPE_DATA_TIME_POINT
					
				//}
				if ((count($arrTP) > 0 && count($arrTR) > 0) ||
						(count($arrTP) > 0 && count($arrRA) > 0))
					throw new Exception(Yii::t('lstrings', 'ERR_FILE_HEADER_FORMAT_COMP'));
				if (count($arrTP) > 0) {
					$type = FileItemFactory::FILE_TYPE_DATA_TIME_POINT;
				} else
					$type = FileItemFactory::FILE_TYPE_DATA_TR_RA;
				/* //get the number of columns
				$columns = count($record); */
				if ($checkDuplicates) {
					$genes = array();
				}
				$lines = 0;
				//count the number of lines in the file; maybe there is a faster  method
				while (($record = fgetcsv($fp, 0, $delim)) !== FALSE) {
					$lines++;
					//search for duplicates, and throw error if any
					if ($checkDuplicates) {
						if (array_key_exists($record[0], $genes))
							throw new Exception(Yii::t('lstrings', '10006').$record[0].'/'.
									$genes[$record[0]].', '.$lines, 10006);
						else
							$genes[$record[0]] = $lines;
					}
					//check if the file has NA values and set the flag accordingly
					if (!$hasNA)
						foreach ($record as $row)
							if (strlen($row) === 0) {
								$hasNA = true;
								break;
							}
				}
				
				fclose($fp);
				//create the file with the data and return it
				if ($type === FileItemFactory::FILE_TYPE_DATA_TR_RA)
					$result = new FileItemDataTRRA(FileItemFactory::FILE_TYPE_DATA_TR_RA, $diskName, $name, 
						($size !== NULL ? $size : filesize($fullPath.$diskName)), 
						$lines, $tr, $ra, $delim, $hasNA, $arrTR, $arrRA);
				else
					$result =  new FileItemDataTimePoint(FileItemFactory::FILE_TYPE_DATA_TIME_POINT,
							$diskName, $name,
							($size !== NULL ? $size : filesize($fullPath.$diskName)),
							$lines, $delim, $hasNA, $tp, $arrTP);
			}
		}
		//database operations 
		if ($result != NULL && !Yii::app()->user->isGuest) {
			/* if ($id == -1) {//create a new record in the database
				$result->setIsNewRecord(true);
			}  */
				$result->saveFile($id);
		} 
		return $result;
		
	}
	
}
