<?php
/**
 * Class which contains all common data for a certain operation like the input file and the result
 * @author andrei
 *
 */
class Operation //extends CFormModel
		extends CActiveRecord
{
	const RESULT_SUCCESS = "__suc__";
	
	const OP_CLUSTER = 11;
	const OP_ANALYSE = 12;
	const OP_FILTER = 13;
	
	const ALGO_UPGMA = 101;
	const ALGO_KMEANS = 102;
	const ALGO_SOTA = 103;
	
	const ALGO_GENARRAY = 201;
	const ALGO_STABILOGEN = 202;
	const ALGO_SHUTOFF = 203;
	
	const ALGO_FILTER_NA = 301;
	
	const EXP_TIMES_TR = 1001;
	const EXP_TIMES_RA = 1002;
	const EXP_TIMES_TR_RA = 1003;
	/** @final int Files containing just one set of columns, each with its own time; Generated by the C programs **/
	const EXP_TIME_POINT = 1004;
	
	
	/** @var int the id of the object operation to be uniquely identified*/
	public $id;
	/** @var int the type of the executed algorithm*/
	public $algorithmType;
	/** @var int the type of operation executed to get the result*/
	public $operationType;
	/** @var long the date when it started */
	public $date;
	/** @var int what times are considered from the input files; TR, RA or both */
	public $experimentTimes = Operation::EXP_TIME_POINT;
	/** @var String a short description of this operation and how it will end up;
	 * a max of 200 chars are allowed*/
	public $description;
	/** @var String title for the operation; max of 70 characters */
	public $title;
	/** @var Result the result for this operation */
	public $result;
	/** @var Input the input data for this operation */
	public $input;
	/** @var Experiment The "father" experiment in which this operation is included;
	 * Set in the __constructor */
	public $experiment;
    
    public function rules()
    {
    	return array(
    			array('title', 'required'),
    			array('experimentTimes', 'in', 'range'=> array(Operation::EXP_TIMES_TR,
    					Operation::EXP_TIMES_RA,
    					Operation::EXP_TIMES_TR_RA,
    					Operation::EXP_TIME_POINT)),
    			array('title', 'length', 'min' => 1, 'max' => Util::TITLE_LEN),
    			array('description', 'length', 'min' => 0, 'max' => Util::DESC_LEN),
    	);
    }
    
    public function attributeLabels()
    {
    	return array('lblTitle'=>Yii::t('lstrings', 'TITLE'),
				'lblDescription'=>Yii::t('lstrings', 'DESCRIPTION'),
    			'lblExpTimes'=>Yii::t('lstrings', 'EXP_TIMES'));
    }
    /**
     * 
     * @param int $operationType the type of operation chosen by the user from OP_CLUSTER, OP_ANALYSE, OP_FILTER
     * @param int $algorithmType the algorithm run by the user; one of ALGO_UPGMA, ALGO_KMEANS, ALGO_SOTA
     * @param long $date the point in time in seconds when the experiment started
     * @param Input $input an object representing the input for this operation
     * @param Result $result the result wrapper for this operation
     * @param int $experimentTimes what columns from the input files can be used by the user; one of EXP_TIMES_TR , 
     * EXP_TIMES_RA, EXP_TIMES_TR_RA; EXP_TIME_POINT
     * @param String $description a short description for the operation; max length of DESC_LEN
     * @param String $title the title of the description; Generally it is generated automatically
     * by the child classes but it can be modified by the user with a custom value; max length of TITLE_LEN
     * @param int $id the id of the current operation
     * @param Experiment $experiment The "father" experiment in which this operation is included; It defaults right now
     * to the currentExperiment object set on the session when this Operation object is being created
     */
    public function __construct($operationType = Operation::OP_ANALYSE, $algorithmType = Operation::ALGO_STABILOGEN, 
			$date = 0, Input $input = NULL, Result $result = NULL,
    		$experimentTimes = NULL, $description = NULL, $title = NULL, $id = Util::ID_UNDEFINED) {
    	$this->id = $id; 
    	$this->algorithmType = $algorithmType;
    	$this->operationType = $operationType;
    	$this->date = $date;
    	$this->experimentTimes = $experimentTimes; 
    	$this->description = $description;
    	$this->title = $title;
    	$this->result = $result;
    	$this->input = $input;
    	//this is just temporary to identofy uniquely an operation while there is no DB connection
    	if (Yii::app()->user->isGuest)
    		$this->id = $date; 
    	//for now it is a good idea to set the father experiment as the current experiment
    	$this->experiment = Yii::app()->session['currentExperiment'];
    }
  
    /**
     * Basic function which uses a an id and calls the aproproate methods to remove the records from the DB
     * @param int $id The id of the operation being targeted for deletion
     * @throws Exception Error when tried to perform the operation
     */
    public function deleteOperation($id) {
    	try {
	    	//delete the result
	    	$this->result->deleteResult($this->result->getId(), Yii::app()->basePath.$this->experiment->getPath());
	    	//delete the input
	    	$this->input->deleteInput($this->input->getId(), Yii::app()->basePath.$this->experiment->getPath());
	    	$this->deleteByPk($id);
    	} catch (Exception $e) {
    		throw $e;
    	}
    }
	
	public function saveOperation($id) {
			$dbI = Yii::app()->db;
		$arrayCols = array(SqlUtil::OPERATION_EXPERIMENT_FK => $this->experiment->getId(),
							SqlUtil::OPERATION_ALGORITHMTYPE => $this->algorithmType,
							SqlUtil::OPERATION_OPERATIONTYPE => $this->operationType,
							SqlUtil::OPERATION_DATE => $this->date,
							SqlUtil::OPERATION_EXPERIMENTTIMES => $this->experimentTimes,
							SqlUtil::OPERATION_DESCRIPTION => $this->description,
							SqlUtil::OPERATION_TITLE => $this->title);
		//$model = FileItem::model();
		if ($id == Util::ID_UNDEFINED) {
			$fileI = $dbI->createCommand()->insert(SqlUtil::OPERATION_TBL,
					$arrayCols);
			$this->id = $dbI->getLastInsertID();
			//save input
			$this->input->saveInput(Util::ID_UNDEFINED, $this->id);
			$this->result->saveResult(Util::ID_UNDEFINED, $this->id);
			/* $model->setIsNewRecord(true);
				$model->save(); */
		} else {
			$fileI = $dbI->createCommand()->update(SqlUtil::OPERATION_TBL,
					$arrayCols,	SqlUtil::ID_COL.' = '.$id);
			$this->id = $id;
			
			$this->input->saveInput($this->input->getId(), $this->id);
			$this->result->saveResult($this->result->getId(), $this->id);
		}
	}
	
	/**
	 * Set the values of the fields if the current class is Operation, else (for childrens)  load the values 
	 * from DB; It is not necessary to reload data from DB in case of Operation since it is already available
	 * from the experiment id search
	 * @param array $rowOperation The result returned by the query which searches all operations for
	 * a certain experiment
	 */
	public function loadOperation($rowOperation) {
		$this->id = $rowOperation[SqlUtil::ID_COL];
		$this->algorithmType = $rowOperation[SqlUtil::OPERATION_ALGORITHMTYPE];
		$this->operationType = $rowOperation[SqlUtil::OPERATION_OPERATIONTYPE];
		$this->date = $rowOperation[SqlUtil::OPERATION_DATE];
		$this->experimentTimes = $rowOperation[SqlUtil::OPERATION_EXPERIMENTTIMES];
		$this->description = $rowOperation[SqlUtil::OPERATION_DESCRIPTION];
		$this->title = $rowOperation[SqlUtil::OPERATION_TITLE];
		try {
			if ($this->result == NULL) {
				$this->result = new Result();
				$this->result->loadResult($this->id);
			}
			if ($this->input == NULL) {
				$this->input = new Input();
				$this->input->loadInput($this->id);
			}
		} catch (Exception $e) {
			throw $e;
		}
	}
	/**
	 * Tries to load all operations associated with a certain experiment 
	 * @param int $idExperiment The id of the experiment which contains all the operations being loaded
	 * @throws Exception
	 */
	public static function loadOperations($idExperiment) {
		
		try {
			$dataR = Yii::app()->db->createCommand()->select()
				->from(SqlUtil::OPERATION_TBL)
				->order(SqlUtil::OPERATION_DATE.' DESC')
				->where(array('and', SqlUtil::OPERATION_EXPERIMENT_FK.'='.$idExperiment))
				->queryAll();
			/* if($dataR == false)
				throw new Exception(Yii::t('lstrings', '10001'), 10001);
			else { */
				$result = array();
				if (count($dataR) > 0) {
						foreach ($dataR as $row) {
							$algoType = $row[SqlUtil::OPERATION_ALGORITHMTYPE];
							$op = NULL;
							switch ($algoType) {
								case Operation::ALGO_FILTER_NA:
									$op = new FilterNAForm();
									break;
								case Operation::ALGO_GENARRAY:
									$op = new GenarrayForm();
									break;
								case Operation::ALGO_STABILOGEN:
									$op = new StabilogenForm();
									break;
								case Operation::ALGO_KMEANS:
									$op = new KmeansForm();
									break;
								case Operation::ALGO_SOTA:
									$op = new SotaForm();
									break;
								case Operation::ALGO_UPGMA:
									$op = new UpgmaForm();
									break;
							}
							if ($op != NULL) {
								$op->loadOperation($row);
								$result[] = $op;
							}
						}
					
				}
				return $result;/* 
			} */
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
	
	public function tableName()
	{
		return SqlUtil::OPERATION_TBL;
	}
	
	public function primaryKey()
	{
		return SqlUtil::ID_COL;
		// For composite primary key, return an array like the following
		// return array('pk1', 'pk2');
	}
	
	
	/**
	 * This method must be implemented by all the classes which inherits the current one
	 * to be able to run the script; It is basically called by the controller and in the end assembles
	 * all the variables and calls executeR;
	 */
	public function runMethod(){}
	
	/**
	 * Use it when you need the name of the current operation; Values are taken from $operationType
	 * field and the corresponding name is returned
	 * @return string the name of the operation, translated in the current language if appropriate
	 */
	public function getOpName() {
		switch ($this->operationType) {
			case Operation::OP_CLUSTER:
				return Yii::t('lstrings', 'CLUSTER_MENU');
			case Operation::OP_ANALYSE:
				return Yii::t('lstrings', 'ANALYSE_MENU');
			case Operation::OP_FILTER:
				return Yii::t('lstrings', 'FILTER_MENU');
		}
	}
	
	/**
	 * @return string controller path for the static view containing the list
	 * of all algorithms in one category
	 */
	public function getOpListLink(){}
	    
	/**
	 * @return string the name of the algorithm
	 */
	public function getAlgoName(){}
	
	/**
	 * @return string the path for the action which will be run by the corresponding
	 * controller
	 */
	public function getAlgoLink(){}
	
	public function getNoInputMsg() {
		return Yii::t('lstrings', 'MSG_MISSING_INPUT');
	}
	
	/**
	 * General method which determines if the warning message about missing values and the filter link
	 * should be displayed
	 * @return boolean Should the message be displayed in the current's model page? It defaults to yes
	 * since the majority of the pages will need it
	 */
	public function showFilterNAReminder() {
		if ($this->experiment->getCurrentOpFile()->getHasNA())
			return true;
		else
			return false;
	}
	
	/**
	 * @return Experiment The "father" experiment of this operation; For now it is the one set in the
	 * session as 'currentExperiment' when the operation was created; the value is set in the __constructor
	 */
	public function getExperiment() {
		return $this->experiment;
	}
	
	/**
	 * Check if the input file has more than one type of expereminet times
	 * @return boolean Show or not the control associated with the selection of the array
	 * experiment times in the GUI
	 */
	public function isSelectExperimentTimesVisible() {
		return $this->experiment->getCurrentOpFile()->isMultiCols();
	}
	
	/**
	 *  @return boolean Display or not the radio buttons' list where the user is allowed to
	 *  select which cols to use for the current operation (between RA and TR); displayed
	 *  only when the input file has both columns and it has the FileItemDataTRRA type
	 */
	public function isRLExpTimesVisible() {
		if ($this->experiment->getCurrentOpFile()->hasTRRA()) {
			if ($this->experiment->getCurrentOpFile()->getNumRA() > 0 &&
					$this->experiment->getCurrentOpFile()->getNumTR() > 0)
				return true;
			else
				return false;
		} else
			return false;
	}
	
	/**
	 * @return array The list of allowed experiment times, when the current file is TR and RA; used
	 * to display the radio button list with the options
	 */
	public function getRLExpTimesList() {
		return array(Operation::EXP_TIMES_TR => Yii::t('lstrings', 'USE_TR'),
        						Operation::EXP_TIMES_RA => Yii::t('lstrings', 'USE_RA'),
        						Operation::EXP_TIMES_TR_RA => Yii::t('lstrings', 'USE_TR_RA'));
	}
	
	public function getHelpField($idHelp) {
		$imghtml = CHtml::image('images/info.png');
		return CHtml::link(CHtml::image('images/info.png'), array('site/helpd', 'id'=>$idHelp));
	}
	
	/**
	 * Helper function to hide the inner workings of the system to determine if the operation
	 * is of a type or another
	 */
	public function isOpClustering() {
		return false;
	}
	
	/**
	 * @return boolean Returns whether an operation is of type analyse
	 */
	public function isOpAnalyse() {
		return false;
	}
	
	public function getId() 
	{
	  return $this->id;
	}
	
	public function setId($value) 
	{
	  $this->id = $value;
	}
	
	public function getAlgorithmType()
	{
		return $this->algorithmType;
	}
	
	public function setAlgorithmType($value)
	{
		$this->algorithmType = $value;
	}
	
	public function getOperationType()
	{
		return $this->operationType;
	}
	
	public function setOperationType($value)
	{
		$this->operationType = $value;
	}	
	
	public function getDate()
	{
		return $this->date;
	}
	
	public function setDate($value)
	{
		$this->date = $value;
	}
	
	public function getDescription()
	{
		return $this->description;
	}
	
	public function setDescription($value)
	{
		$this->description = $value;
	}

	public function getTitle()
	{
		return $this->title;
	}
	
	public function setTitle($value)
	{
		$this->title = $value;
	}
	
	public function getResult()
	{
		return $this->result;
	}
	
	public function setResult($value)
	{
		$this->result = $value;
	}
	
	public function getInput()
	{
		return $this->input;
	}
	
	public function setInput($value)
	{
		$this->input = $value;
	}
	
	public function getExperimentTimes()
	{
		if ($this->experiment->getCurrentOpFile()->hasTRRA()) {
			if ($this->experiment->getCurrentOpFile()->getNumRA() > 0
					&& $this->experiment->getCurrentOpFile()->getNumTR() > 0)
			return $this->experimentTimes;
			else if ($this->experiment->getCurrentOpFile()->getNumRA() > 0)
				return Operation::EXP_TIMES_RA;
			else if ($this->experiment->getCurrentOpFile()->getNumTR() > 0)
				return Operation::EXP_TIMES_TR;
		} else if ($this->experiment->getCurrentOpFile()->hasTimePoint()) {
			return Operation::EXP_TIME_POINT;
		}
	}
	
	public function setExperimentTimes($value)
	{
		$this->experimentTimes = $value;
	}
	
	/**
	 * Load all necessary parameters in a new object; The real destination of this method is to clone the old
	 * values (visible in the algorithm's form) set by the user when he/she run the operation; next time when the user returns to the form
	 * he/she will get the latest values he/she set; 
	 * @param Operation $newObject The new object for which certain attributes will be set
	 */
	public function loadOldValues(Operation $oldObject) {
		$this->title = $oldObject->getTitle();
		$this->description = $oldObject->getDescription();
		$this->experimentTimes = $oldObject->getExperimentTimes();
		$this->experiment = $oldObject->getExperiment();		
	}
	
	
}