<?php
/**
 * Base class for all clustering algorithm implementing all common methods
 * @author andrei
 *
 */
class ClusterForm extends Operation
{
	
	const FILE_CLUSTER_RESULT_EXT = '.csv'; 
	const FILE_CLUSTER_DETAILS_EXT = '.clu';
	
	const VOUT_TYPE_HEATMAP = 1101;
	const VOUT_TYPE_DENDROGRAM = 1102;
	const VOUT_TYPE_CLUSTER = 1103;
	
	const VOUT_FORMAT_SVG = '.svg';
	const VOUT_FORMAT_JPG = '.jpg';
	const VOUT_FORMAT_PNG = '.png';
	
	/** @var int Don't use a normalization method for the column values **/ 
	const NORM_METHOD_NONE = 1201;
	/** @var int Divide each column by the first one (first one divided by the first one will be 1.00) **/
	const NORM_METHOD_DIV = 1202;	
	/** @var int Divide each column by the first one (first one divided by the first one will be 1.00) and
	 * in the end use the log2 of the result **/
	const NORM_METHOD_LOG_DIV = 1203;
	/** @var int Apply the log2 over the unmodified values of each column **/
	const NORM_METHOD_LOG = 1204;
	
	/** @var int Run the normalization method after the data has been read from the file but before doing any clustering or displaying (normalized data will be clustered and then displayed) */
	const NORM_METHOD_WHEN_BEF_CL = 1301;
	/** @var int Run the normalization method after the data has been read from the file and after running the clustering method, but before displaying */
	const NORM_METHOD_WHEN_AFT_CL = 1302;
	
	/** @const int Colors starting from green (lowest) and ending with red (highest values)*/
	const HEATMAP_COL_GREEN_RED = 1401;
	/** @const int Colors starting from yellow (lowest) and ending with blue (highest values) */
	const HEATMAP_COL_YELLOW_BLUE = 1402;
	
	public $voutType = ClusterForm::VOUT_TYPE_HEATMAP;
	/** @var int flag representing the format of the cluster result drawing; can take one of the VOUT_FORMAT_* vals */ 
	public $voutFormat = ClusterForm::VOUT_FORMAT_SVG;
	/**  @var int Normalization method used before the actual clustering operational for all the experiment times in the file (it doesn't matter if it is a RA or TR column, it will be normalized against the first column of each type of experiment) */
	public $normMethod = ClusterForm::NORM_METHOD_NONE;
	/**  @var boolean Flag which decides if the current operation file will be replaced with the file generated by clustering */
	public $replaceWithClRes = '0';
	/**  @var int Choose when to apply the normalization in the R clustering files. Possible values are NORM_METHOD_WHEN_BEF_CL and NORM_METHOD_WHEN_AFT_CL */
	public $whenApplyNorm = ClusterForm::NORM_METHOD_WHEN_BEF_CL;
	/**  @var int The type of the color scale to be used when drawing the bitmap */
	public $heatmapCol = ClusterForm::HEATMAP_COL_GREEN_RED;
	
	public function __construct($operationType = ClusterForm::OP_CLUSTER, 
			$algorithmType = Util::ID_UNDEFINED, $date = 0, Input $input = NULL, Result $result = NULL,
			$experimentTimes = NULL, $description = NULL, $title = NULL, $id = Util::ID_UNDEFINED) {
		parent::__construct($operationType, $algorithmType, $date,  $input, $result,
				$experimentTimes, $description, $title, $id);
	}
	
	public function primaryKey()
	{
		return SqlUtil::CLUSTERFORM_OPERATION_FK;
	}
	
	public function deleteOperation($id) {
		try {
			$this->deleteByPk($id);
			parent::deleteOperation($id);
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function loadOperation($rowOperation) {
		$dataR = Yii::app()->db->createCommand() ->select()
		->from(SqlUtil::CLUSTERFORM_TBL)
		->where(array('and', SqlUtil::CLUSTERFORM_OPERATION_FK.'='.$rowOperation[SqlUtil::ID_COL]))
		->queryRow();
		if($dataR == false)
			throw new Exception(Yii::t('lstrings', '10001'), 10001);
		else {
			$this->voutType = $dataR[SqlUtil::CLUSTERFORM_VOUTTYPE];
			$this->voutFormat = $dataR[SqlUtil::CLUSTERFORM_VOUTFORMAT];
			$this->normMethod = $dataR[SqlUtil::CLUSTERFORM_NORMMETHOD];
			$this->replaceWithClRes = $dataR[SqlUtil::CLUSTERFORM_REPLACEWITHCLRES];
			$this->whenApplyNorm = $dataR[SqlUtil::CLUSTERFORM_WHENAPPLYNORM];
			$this->heatmapCol = $dataR[SqlUtil::CLUSTERFORM_HEATMAPCOL];
			try {
				parent::loadOperation($rowOperation);
			}  catch (Exception $e) {
				throw $e;
			}
		}
	}
	
	public function saveOperation($id) {
		parent::saveOperation($id);
		$fileI = Yii::app()->db->createCommand()->insert(SqlUtil::CLUSTERFORM_TBL,
				array(SqlUtil::CLUSTERFORM_OPERATION_FK => $this->id,
						SqlUtil::CLUSTERFORM_VOUTTYPE => $this->voutType,
						SqlUtil::CLUSTERFORM_VOUTFORMAT => $this->voutFormat,
						SqlUtil::CLUSTERFORM_NORMMETHOD => $this->normMethod,
						SqlUtil::CLUSTERFORM_REPLACEWITHCLRES => $this->replaceWithClRes,
						SqlUtil::CLUSTERFORM_WHENAPPLYNORM => $this->whenApplyNorm,
						SqlUtil::CLUSTERFORM_WHENAPPLYNORM => $this->heatmapCol));
	}
	
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
	
	public function tableName()
	{
		return SqlUtil::CLUSTERFORM_TBL;
	}
	
	
	public function rules()
	{
		return array_merge(parent::rules(),
			array(array('normMethod', 'in', 'range'=> array(ClusterForm::NORM_METHOD_NONE, 
					ClusterForm::NORM_METHOD_DIV, ClusterForm::NORM_METHOD_LOG_DIV,
					ClusterForm::NORM_METHOD_LOG)),
				array('whenApplyNorm', 'in', 'range'=> array(ClusterForm::NORM_METHOD_WHEN_BEF_CL,
						ClusterForm::NORM_METHOD_WHEN_AFT_CL)),
				array('voutType', 'in', 'range'=> array(ClusterForm::VOUT_TYPE_HEATMAP, 
					ClusterForm::VOUT_TYPE_DENDROGRAM, ClusterForm::VOUT_TYPE_CLUSTER)),
				array('replaceWithClRes', 'numerical', 'max' => 1, 'min' => 0, 'integerOnly'=>true),
				array('voutFormat', 'in', 'range'=> array(ClusterForm::VOUT_FORMAT_SVG, 
					ClusterForm::VOUT_FORMAT_JPG, ClusterForm::VOUT_FORMAT_PNG)),
				array('heatmapCol', 'in', 'range'=> array(ClusterForm::HEATMAP_COL_GREEN_RED, 
					ClusterForm::HEATMAP_COL_YELLOW_BLUE))));
	}
	
	public function attributeLabels()
	{
		return array_merge(parent::attributeLabels(), 
				array('lblNormMethod' => Yii::t('lstrings', 'CL_NORM_METHOD_LBL'),
						'lblWhenApplyNorm' => Yii::t('lstrings', 'CL_NORM_WHEN_LBL'),
						'lblVoutType' => Yii::t('lstrings', 'CL_VIS_OUT_TYPE'),
						'lblReplaceWithClRes' => Yii::t('lstrings', 'CL_REPL_CL_RES'),
						'lblVoutFormat' => Yii::t('lstrings', 'CL_VIS_OUT_FORMAT'),
						'lblHeatmapCol' => Yii::t('lstrings', 'CL_HEATMAP_COL')));
	}
	
	/**
	 * This function is used to execute a R command and send the required and
	 * specific parameters from each type of algorithm 
	 * @param String $arguments
	 */
	protected function executeR($script, $arguments) {
		//create path in which to save experiment data
		//run the script	 
		$nameOutput = $this->date.$this->getAlgoName();
		$ret = shell_exec('Rscript '. Yii::app()->basePath.'/commands/shell/r/'.$script.' '.				
				Yii::app()->basePath.
				$this->experiment->path.' '.
				$this->experiment->getCurrentOpFile()->getDiskName().' '.
				$nameOutput.' '.
				Operation::RESULT_SUCCESS.' '.
				$this->getExperimentTimes().' '.
				$this->experiment->getCurrentOpFile()->getDelimiter().' '.
				$this->voutType.' '.
				$this->voutFormat.' '.
				$this->normMethod.' '.
				$this->whenApplyNorm.' '.
				$this->heatmapCol.' '.
				$arguments.
				' 2>&1');
		//echo $ret;
		$path = Yii::app()->basePath.
				$this->experiment->path.$nameOutput;
		//check if the execution was successfull by examining the output, and the 2 required out files
		#echo $ret;
		if (strpos($ret, Operation::RESULT_SUCCESS) === false ||
			!file_exists($path.$this->voutFormat) /*|| 				
				 (((int)$this->voutType) !== ClusterForm::VOUT_TYPE_HEATMAP && !file_exists($path.ClusterForm::FILE_CLUSTER_RESULT_EXT)) */
			) {
			$this->result->setMessage($ret);
		} else {
			$this->input->setFile($this->experiment->getCurrentOpFile());
			$this->result->setMessage(Operation::RESULT_SUCCESS);
			
				$this->result->setFile(FileItemFactory::createFile(FileItemFactory::FILE_TYPE_GENERIC, 
					Yii::app()->basePath.$this->experiment->path, 
						$nameOutput.$this->voutFormat,
						$this->getAlgoName().$this->voutFormat));
				//if (((int)$this->voutType) !== ClusterForm::VOUT_TYPE_HEATMAP) {//in case of heatmap we don't have a new file
					//create the result file containing the clusters along with the mean values for each column

				// check if the current settings of the algorithm supports output as a CSV file with clusters and their values for each experiment time
				if ($this->hasResultFileData()) {
					$fileData = FileItemFactory::createFile($this->input->getFile()->getFileType(),
							Yii::app()->basePath.$this->experiment->path,
							$nameOutput.ClusterForm::FILE_CLUSTER_RESULT_EXT,
							$this->getAlgoName().ClusterForm::FILE_CLUSTER_RESULT_EXT);
					//save the result file with the clusters in the result object of the operation
					$this->result->setFileData($fileData);
					//replace the current input if the user wants that, and add the new file instead
					if ($this->replaceWithClRes === '1')
						$this->experiment->setCurrentOpFile($fileData);
				}
					
					
					
					if ($this->hasResultFileCluster()) {
						//create the file containing the cluster details (name/number of genes/list of genes)
						$this->result->setFileCluster(FileItemFactory::createFile(FileItemFactory::FILE_TYPE_GENERIC,
								Yii::app()->basePath.$this->experiment->path,
								$nameOutput.ClusterForm::FILE_CLUSTER_DETAILS_EXT,
								$this->getAlgoName().ClusterForm::FILE_CLUSTER_DETAILS_EXT));
					}
					
				//}
			$this->experiment->addOperationHistory($this);
			//if user has logged in save the result in the DB
			if (!Yii::app()->user->isGuest) {
				$this->saveOperation(Util::ID_UNDEFINED);
			}	
		}
		//echo $ret;
	}
	
	public function isOpClustering() {
		return true;
	}
	
	public function getOpListLink() {
		return 'exp/experiment/page/view/cluster_list';
	}
	
	public function getOpName() {
		return Yii::t('lstrings', 'CLUSTER_MENU');
	}
	
	/**
	 * Wrapper function to generate the result for the clustering method
	 * @return mixed either an SVG image if the execution was successfull or 
	 * a string representing the error of the R program
	 * (non-PHPdoc)
	 * @see Operation::getResult()
	 */
	public function getResultView() {
		if ($this->result->getMessage() === Operation::RESULT_SUCCESS) {
			return /* file_get_contents(Yii::app()->basePath.
					$this->experiment->path.
					$this->result->getFile()->getDiskName()); */
			'<img src="/exp/op/cluster/getclusterimg" 
				alt="'. Yii::t('lstrings', 'CLUSTER_MENU').'" 
				/>';
		} else {
			return $this->result->getMessage();
		}
	}
	
	/**
	 * Each type of clustering has its own list of diplaying the results; return the right list
	 * for each algorithm; since the majority of algos can produce a dendogram and a heatmap
	 * the standard implementation will return an array of those 2
	 * @return array the array of key/value pairs which will be displayed to the user 
	 */
	public function getArrayVisualOutType() { 
		return array(ClusterForm::VOUT_TYPE_HEATMAP => Yii::t('lstrings', 'CL_VIS_OUT_TYPE_HEA'),
				ClusterForm::VOUT_TYPE_DENDROGRAM => Yii::t('lstrings', 'CL_VIS_OUT_TYPE_DEN'));
	}
	
	/**
	 * @return array the array of key/value pairs which will be displayed to the user containing the
	 * types of formats allowed in the application; this is a wrapper for the values VOUT_FORMAT_*
	 */
	public function getArrayVisualOutFormat() {
		return array(ClusterForm::VOUT_FORMAT_SVG => 'SVG',
				ClusterForm::VOUT_FORMAT_JPG => 'JPEG',
				ClusterForm::VOUT_FORMAT_PNG => 'PNG');
	}
	
	public function getArrayHeatmapCols() {
		return array(ClusterForm::HEATMAP_COL_GREEN_RED => Yii::t('lstrings', 'CL_HEATMAP_COL_GR'),
				ClusterForm::HEATMAP_COL_YELLOW_BLUE => Yii::t('lstrings', 'CL_HEATMAP_COL_YB'));
	}
	
	
	/**
	 * Check if the control for selecting when to apply the normalization is disabled or not. Normally when any type of normalization is active, this control must be enabled. it is disabled for $this->normMethod === NORM_METHOD_NONE.
	 */
	public function isWhenApplyNormDisabled() {
		return $this->normMethod === ClusterForm::NORM_METHOD_NONE;
	}
	
	/**
	 * Check if the algorithm will generate an output with clusters details (Cluster name, number of genes and list of the genes)
	 * @return boolean Always true for Kmeans and Sota; UPGMA only returns a file when the dendrogram is cut
	 */
	public function hasResultFileCluster() {
		if ($this->result->getMessage() === Operation::RESULT_SUCCESS) 
			return true;
		else 
			return false;
	}
	
	/**
	 * @return boolean The current settings for the algorithm support saving the result as a clustering file
	 */
	public function hasResultFileData() {
		if ($this->result->getMessage() === Operation::RESULT_SUCCESS) 
			return true;
		else 
			return false;
	}
	
	/**
	 * @return boolean Check whether the control for selecting the color of the heatmap is disabled or not
	 * It must be disabled when the selected output is something else than a heatmap
	 */
	public function isHeatmapColDisabled() {
		return $this->voutType != ClusterForm::VOUT_TYPE_HEATMAP;
	}
	
	public function loadOldValues(Operation $oldObject) {
		parent::loadOldValues($oldObject);
		$this->heatmapCol = $oldObject->getHeatmapCol();
		$this->voutType = $oldObject->getVoutType();
		$this->voutFormat = $oldObject->getVoutFormat();
		$this->normMethod = $oldObject->getNormMethod();
		$this->replaceWithClRes = $oldObject->getReplaceWithClRes();
	}
	
	public function getHeatmapCol() {
		return $this->heatmapCol;
	}
	
	public function getVoutType() {
		return $this->voutType;
	}
	
	public function getVoutFormat() {
		return $this->voutFormat;
	}
	
	public function getNormMethod() {
		return $this->normMethod;
	}
	
	public function getReplaceWithClRes() {
		return $this->replaceWithClRes;
	}
	
}