<?php
/**
 * Class containing all methods and attributes necessary to run the Kmeans algorithm
 * @author andrei
 *
 */
class KmeansForm extends ClusterForm
{
	
	const TYPE_CENTER_TR = 101;
	const TYPE_CENTER_RA = 101;
	const TYPE_CENTER_TIMEPOINT = 101;
	
	/** @const int Display the TR RA as one big matrix where the rows and columns are obtained by merging TR and RA arrays of times */
	const KMEANS_CLUSTER_SAME_MAT = 1501;
	/** @const int Display the TR RA as 2 different matrices with rows and columns respresented by the times of TR and/or RA respectevely */
	const KMEANS_CLUSTER_DIFF_MAT = 1502;
	
	/** @var array The list of centers for TR */
	public $centersTR;
	/** @var array The list of centers for RA */
	public $centersRA;
	/** @var array The list of centers for Time point */
	public $centersTimePoint;
	/** @var int Number of clusters */
	public $clusters;
	/** @var int How many iterations should the algorithm actually do? It may need to be fine tuned to find
	 * the actual convergence point */
	public $maxNumIterations = 10;
	/** @var int The number of random sets to be generated */
	public $numRandSets = 1;
	/** @var string The method used to get the distances */
	public $distMethod;
	/** @var int Selects the display method of the TR and RA columns; It is valid only when the display is a cluster, not a dendrogram and the file contains both TR and RA values; It takes one of the following values: KMEANS_CLUSTER_SAME_MAT or KMEANS_CLUSTER_DIFF_MAT */
	public $clusterDisplay = KmeansForm::KMEANS_CLUSTER_DIFF_MAT;
	
	public function __construct($operationType = KmeansForm::OP_CLUSTER, 
				$algorithmType = KmeansForm::ALGO_KMEANS, $date = 0, Input $input = NULL, 
				Result $result = NULL, $title = NULL, $description = NULL, 
				$experimentTimes = Operation::EXP_TIMES_TR_RA, $id = Util::ID_UNDEFINED) {
		parent::__construct($operationType, $algorithmType, $date,  $input, $result,
    		$experimentTimes, $description, $title, $id);
	}
	
	public function primaryKey()
	{
		return SqlUtil::KMEANSFORM_CLUSTERFORM_FK;
	}
	
	public function deleteOperation($id) {
		try {
			$this->deleteByPk($id);
			parent::deleteOperation($id);
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function loadOperation($rowOperation) {
		$dataR = Yii::app()->db->createCommand() ->select()
			->from(SqlUtil::KMEANSFORM_TBL)
			->where(array('and', SqlUtil::KMEANSFORM_CLUSTERFORM_FK.'='.$rowOperation[SqlUtil::ID_COL]))
			->queryRow();
		if($dataR == false)
			throw new Exception(Yii::t('lstrings', '10001'), 10001);
		else {
			$this->clusters = $dataR[SqlUtil::KMEANSFORM_CLUSTERS];
			$this->maxNumIterations = $dataR[SqlUtil::KMEANSFORM_MAXNUMITERATIONS];
			$this->numRandSets = $dataR[SqlUtil::KMEANSFORM_NUMRANDSETS];
			$this->distMethod = $dataR[SqlUtil::KMEANSFORM_DISTMETHOD];	
			$this->clusterDisplay = $dataR[SqlUtil::KMEANSFORM_CLUSTERDISPLAY];		
					
			try {
				$this->centersTR = $this->loadArrayCenters(KmeansForm::TYPE_CENTER_TR, $rowOperation[SqlUtil::ID_COL]);
				$this->centersRA = $this->loadArrayCenters(KmeansForm::TYPE_CENTER_RA, $rowOperation[SqlUtil::ID_COL]);
				$this->centersTimePoint = $this->loadArrayCenters(KmeansForm::TYPE_CENTER_TIMEPOINT, $rowOperation[SqlUtil::ID_COL]);
				$this->result = new ResultCluster();
				$this->result->loadResult($rowOperation[SqlUtil::ID_COL]);
				parent::loadOperation($rowOperation);
			}  catch (Exception $e) {
				throw $e;
			}
		}
	}
	
	/**
	 * Loads the values from the kmeans array database 
	 * @param int $centerType The type of the centers to look for
	 * @param int $idKmeans The identifier for main record in the kmeans table to which the searched
	 * centers are associated with
	 */
	private function loadArrayCenters($centerType, $idKmeans) {
		$dataR =  Yii::app()->db->createCommand() ->select()
		->from(SqlUtil::KMEANSCENTERS_TBL)
		->where(array('and', SqlUtil::KMEANSCENTERS_KMEANSFORM_FK.'='.$idKmeans),
				array('and', SqlUtil::KMEANSCENTERS_TYPE.'='.$centerType))
		
		->queryAll();
		if($dataR == false)
			return NULL;
		else {
			$result = array();
			$index = 0;
			foreach ($dataR as $row) {
				$result[$index++] = $row[SqlUtil::KMEANSCENTERS_VALUE];
			}
			return count($result) == 0 ? NULL : $result;
		}
	}
	
	public function saveOperation($id) {
		parent::saveOperation($id);
		$fileI = Yii::app()->db->createCommand()->insert(SqlUtil::KMEANSFORM_TBL,
				array(SqlUtil::KMEANSFORM_CLUSTERFORM_FK => $this->id,
						SqlUtil::KMEANSFORM_CLUSTERS => $this->clusters,
						SqlUtil::KMEANSFORM_MAXNUMITERATIONS => $this->maxNumIterations,
						SqlUtil::KMEANSFORM_NUMRANDSETS => $this->numRandSets,
						SqlUtil::KMEANSFORM_DISTMETHOD => $this->distMethod,
						SqlUtil::KMEANSFORM_CLUSTERDISPLAY => $this->clusterDisplay));
		if ($this->centersTR != NULL)
			$this->insertArrayDb($this->centersTR, $this->id, KmeansForm::TYPE_CENTER_TR);
		if ($this->centersRA != NULL)
			$this->insertArrayDb($this->centersRA, $this->id, KmeansForm::TYPE_CENTER_RA);
		if ($this->centersTimePoint != NULL)
			$this->insertArrayDb($this->centersTimePoint, $this->id, KmeansForm::TYPE_CENTER_TIMEPOINT);
		
	}
	
	/**
	 * Inserts an array in a certain table using a transaction for fast processing of data.
	 * @param array $arrayValues The list of the values to be inserted
	 * @param int $id The id of the KmeansForm record newly inserted
	 * @param int $type Type of the 
	 * @throws Exception Unable to perform the required operation, rollback everything
	 */
	private function insertArrayDb(array $arrayValues, $id, $type) {
		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			foreach ($arrayValues as $key=>$value) {
				$dbI->createCommand()->insert(SqlUtil::KMEANSCENTERS_TBL, 
						array(SqlUtil::KMEANSCENTERS_KMEANSFORM_FK => $id,
								SqlUtil::KMEANSCENTERS_VALUE => $value,
								SqlUtil::KMEANSCENTERS_TYPE => $type));
			}
			$transaction->commit();
		}
		catch (Exception $e)
		{
			$transaction->rollBack();
			//Yii::app()->user->setFlash('error', "{$e->getMessage()}");
			$this->refresh();
			throw $e;
		}
	}
	
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
	
	public function tableName()
	{
		return SqlUtil::KMEANSFORM_TBL;
	}
	
	/**
	 * Declares attribute labels.
	 */
	public function attributeLabels()
	{
		return array_merge(parent::attributeLabels(), array(
				'lblCenters'=>Yii::t('lstrings', 'KMEANS_NUM_CENTERS'),
				'lblClusters'=>Yii::t('lstrings', 'KMEANS_NUM_CLUSTERS'),
				'maxNumIterations'=>Yii::t('lstrings', 'KMEANS_MAX_NUM_ITER'),
				'numRandSets'=>Yii::t('lstrings', 'KMEANS_NUM_RANDOM_SETS'),
				'distMethod'=>Yii::t('lstrings', 'KMEANS_DISTANCE_METHOD'),
				'lstCenters'=>Yii::t('lstrings', 'KMEANS_ADD_CENTS_VALS'),
				'lstClusters'=>Yii::t('lstrings', 'KMEANS_ADD_CLUSTS_VALS'),
				'lstClusterDisplay'=>Yii::t('lstrings', 'KMEANS_CLUST_DISP')
		));
	}
	
	public function rules()
	{
		return array_merge(parent::rules(), array(
				/* array('clusters', 'default', 
					'value'=>Yii::app()->session['currentExperiment']->getCurrentOpFile()->getNumGenes()), */
				array('centersTR', 'safe'),
				array('centersRA', 'safe'),
				array('centersTimePoint', 'safe'),
				array('maxNumIterations, distMethod', 'required'),
				array('maxNumIterations', 'numerical', 'integerOnly'=>true),
				/* array('centers', 'safe'),//, 'on' =>'centers'), */
				array('clusters', 'safe'), /*'numerical', 'integerOnly'=> true, 'allowEmpty' => true,
						'min'=> 1, 'max' => 
						Yii::app()->session['currentExperiment']->getCurrentOpFile()->getNumGenes()) */// 'required', 'on' =>'clusters'),
				/* array('centers', 'arrayLen', 'len'=>
						Yii::app()->session['currentExperiment']->getCurrentOpFile() !== NULL ?
						Yii::app()->session['currentExperiment']->getCurrentOpFile()->getNumCols() : 0),  */
				array('numRandSets', 'safe'),// 'required', 'on' =>'clusters'),
				array('distMethod', 'in', 'range'=> array('euclidean', 'maximum', 'manhattan',
						'canberra', 'binary', 'pearson', 'abspearson', 'correlation',
						'abscorrelation', 'spearman', 'kendall')),
				array('clusterDisplay', 'in', 'range'=> array(KmeansForm::KMEANS_CLUSTER_DIFF_MAT, 
					KmeansForm::KMEANS_CLUSTER_SAME_MAT)),
				
		));
	}
	
	public function runMethod() {
		//check if the user wants to provide a number of clusters
		//or a list of centers
		if (isset($this->clusters)) {
			$firstField = $this->clusters;
		} else {
			if (isset($this->centersTR) && isset($this->centersRA)) {
				$firstField = implode('_', array_merge($this->centersTR, $this->centersRA));
			} else if (isset($this->centersTR)) {
				$firstField = implode('_', $this->centersTR);
			} else if (isset($this->centersRA)) {
				$firstField = implode('_', $this->centersRA);
			} else if (isset($this->centersTimePoint)) {
				$firstField = implode('_', $this->centersTimePoint);
			}
		}
		if ($this->clusterDisplay == NULL)
			$this->clusterDisplay = KmeansForm::KMEANS_CLUSTER_SAME_MAT;
		parent::executeR('kmeans.r', $this->clusterDisplay.' '.
				$firstField.
				' '.$this->maxNumIterations.' '.
				$this->distMethod.' '.(isset($this->clusters) ? 
				$this->numRandSets : ''));
	}
	
	/* public function getArrayCentersTR() {
		$arrayTimesTR = Yii::app()->session['currentExperiment']->getCurrentOpFile()->getArrayTimesTR();
		$len = count($arrayTimesTR);
		$result = array();
		for ($index=0; $index<$len; $index++) {
			$result[$index] = 'TR '.$arrayTimesTR[$index];
		}
	} */
	
	/**
	 * Check if the number of centers in the list is the same with the number of columns
	 * in the current target file
	 * @param unknown_type $attribute
	 * @param unknown_type $params
	 */
public function arrayLen($attribute,$params)
{
    if (isset($this->$attribute) && isarray($this->$attribute) && $params['len'] !== count($this->$attribute))
      $this->addError($attribute, Yii::t('lstrings', 'ERR_KMEANS_CENTS_LEN'));
}
	
	public function getAlgoLink() {
		return 'exp/op/cluster/kmeans';
	}
	
	public function getAlgoName() {
		return Yii::t('lstrings', 'KMEANS_SMENU');
	}
	
	public function getArrayVisualOutType() {
		return array(ClusterForm::VOUT_TYPE_HEATMAP => Yii::t('lstrings', 'CL_VIS_OUT_TYPE_HEA'),
				ClusterForm::VOUT_TYPE_CLUSTER => Yii::t('lstrings', 'CL_VIS_OUT_TYPE_CLU'));
	}
	
	/**
	 * Function needed because it is impossible to set the default value from YII
	 * @return int Value needed in HTML by the add number of clusters and random sets option
	 */
	public function getNumGenes() {
		return (int)($this->experiment->getCurrentOpFile()->getNumGenes()/2);
	}
	
	/**
	 * @return boolean Has the input file TR columns? If yes display the controls associated with it 
	 * (in this case select centers list)
	 */
	public function isSetExpTimesTRVisible() {
		if ($this->experiment->getCurrentOpFile()->hasTRRA())
			return $this->experiment->getCurrentOpFile()->getNumTR() > 0;
		else 
			return false;
	}
	
	/**
	 * @return boolean Has the input file RA columns? If yes display the controls associated with it
	 * (in this case select centers list)
	 */
	public function isSetExpTimesRAVisible() {
		if ($this->experiment->getCurrentOpFile()->hasTRRA())
			return $this->experiment->getCurrentOpFile()->getNumRA() > 0;
		else
			return false;
	}
	
	/**
	 *  @return boolean Check the condition of displaying the list in which the user can add the
	 *  cluster centers for FileItemDataTimePoint file type
	 */
	public function isSetExpTimePointVisible() {
		if ($this->experiment->getCurrentOpFile()->hasTimePoint())
			return $this->experiment->getCurrentOpFile()->getNumTimePoint() > 0;
		else
			return false;
	}
	
	public function getClusterDisplayArr() {
		return array(KmeansForm::KMEANS_CLUSTER_DIFF_MAT => Yii::t('lstrings', 'KMEANS_CLUST_DISP_DIFF'),
				KmeansForm::KMEANS_CLUSTER_SAME_MAT => Yii::t('lstrings', 'KMEANS_CLUST_DISP_SAME'));
	}
	
	public function loadOldValues(Operation $oldObject) {
		parent::loadOldValues($oldObject);
		$this->clusters = $oldObject->getClusters();
		$this->maxNumIterations = $oldObject->getMaxNumIterations();
		$this->numRandSets = $oldObject->getNumRandSets();
		$this->distMethod = $oldObject->getDistMethod();
		$this->clusterDisplay = $oldObject->getClusterDisplay();
	}
	
	public function getClusterDisplay() {
		return $this->clusterDisplay;
	}
	
	public function getDistMethod() {
		return $this->distMethod;
	}
	
	public function getNumRandSets() {
		return $this->numRandSets;
	}
	
	public function getMaxNumIterations() {
		return $this->maxNumIterations;
	}
	
	public function getClusters() {
		return $this->clusters;
	}
	
}